/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package rdc.groups;

import rdc.RDCSample;

import com.inet.report.Area;
import com.inet.report.BorderProperties;
import com.inet.report.DatabaseField;
import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.Group;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.SpecifiedOrder;
import com.inet.report.TableSource;

/**
 * This sample shows how to build a report with groups and sort data by specified order. This shows cities names grouped
 * by specified formula. First formula comprises all cities with name starts with "S", second formula all cities ends
 * with "ton" and third the city of London. All other cities will be grouped in "Other".<BR>
 * <BR>
 * The Data Source Configuration "Sample Database" is necessary for this sample. You can create it using the Data Source
 * Manager in i-net Designer or Configuration Manager.
 */
public class SpecificSortOrderSample extends RDCSample {
    static final String[] EMPLOYEE_ADRESSES_NAMES = { "City", "Region", "Country" };
    static final int[]    EMPLOYEE_ADRESSES_TYPES = { Field.STRING, Field.STRING, Field.STRING };

    /**
     * Creates a new report, add database fields, a group and sort data by specified order.
     * @param exportFmt output format
     * @return engine the engine for the specified report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            // Set database connection
            Datasource ds = eng.getDatabaseTables().getDatasource( 0 ); // the default Datasource(the first) always exists. Otherwise call first createDatasource()
            ds.setDataSourceConfigurationName( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Create a table
            TableSource ts_employeeAdrsses = ds.createTableSource( "EmployeeAddresses", "Employee_Addresses" );
            for( int colIdx = 0; colIdx < EMPLOYEE_ADRESSES_NAMES.length; colIdx++ ) {
                ts_employeeAdrsses.addColumn( EMPLOYEE_ADRESSES_NAMES[colIdx], EMPLOYEE_ADRESSES_TYPES[colIdx] );
            }

            // Add a database field to the report
            Fields fields = eng.getFields();
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );

            DatabaseField city = fields.getDatabaseField( "Employee_Addresses.City" );
            dSec.addFieldElement( city, 3000, 100, 2000, 500 );

            // Add a group
            Group groupCity = eng.addGroup( city );
            groupCity.setSort( Group.SORT_SPECIFIED_ORDER );

            // Create specified sort order
            SpecifiedOrder sorder = new SpecifiedOrder();
            groupCity.setSpecifiedOrder( sorder );
            sorder.add( "Cities starts with S", "{Employee_Addresses.City} like \"S*\"" );
            sorder.add( "Cities ends with ton", "{Employee_Addresses.City} like \"*ton\"" );
            sorder.add( "London", "{Employee_Addresses.City}=\"London\"" );
            //sets what happens with other records
            sorder.setOtherType( SpecifiedOrder.OTHER_RECORDS_GROUP_TO_OTHERS );

            // Add a groupname field to group header
            Area gh1Area = groupCity.getHeader();
            Section gh1Sec = gh1Area.getSection( 0 );
            FieldElement gNField1a = gh1Sec.addFieldElement( fields.getGroupNameField( 0 ), 60, 230, 5425, 230 );
            gNField1a.setBold( true );
            gNField1a.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 1 );
            return null;
        }
    }

    /**
     * Main method of the sample.
     * @param args no parameters used
     */
    public static void main( String[] args ) {
        new SpecificSortOrderSample().initUI();
    }
}
